/*Sensor de fuerza. 
El código está adaptado de https://github.com/HeiLaut/ble-physics-sensors/tree/main/Force%20Sensor
bajo una licencia para el original del MIT https://github.com/HeiLaut/ble-physics-sensors/blob/main/LICENSE

Se necesita una celda de carga de 1 kg y el amplificador de señal Hx711.
También un microprocesador esp32 que incorpora bluetooth.

Se han de instalar las librerias HX711_ADC.h (de Olav Kallhovd, para el amplificador de señal) 
y phyphoxBle.h (de RWTH para la comunicación con la aplicación Phyphox via bluetooth).

La celda de carga se ha de conectar al amplificador de señal con el cable rojo a E+, 
el cable negro a E-, el cable blanco a A- y el cable verde a A+
El amplificador de señal se conecta: VCC a 3,3 V, GND a GND, DT al pin 4 y SCK al pin 5.
*/

#include <HX711_ADC.h>
#include <phyphoxBle.h>

#define BUTTON_PIN 2
#define SENSOR_NAME "Sensor de fuerza M"

// Pines del HX711: DT = 4, SCK = 5
HX711_ADC LoadCell(4, 5);

// Factor de calibración (gramos)
const float calFactor = 1119.22;

// Variables de control
unsigned long t_offset = 0;

void setup() {

  Serial.begin(115200);

  // --- Configuración Phyphox ---
  PhyphoxBLE::start(SENSOR_NAME);
  PhyphoxBLE::experimentEventHandler = &newExperimentEvent;
  PhyphoxBLE::printXML(&Serial);

  PhyphoxBleExperiment exp;
  exp.setTitle("Sensor de fuerza");
  exp.setCategory("Arduino Experiments");
  exp.setDescription("Dibuja la gráfica de la fuerza y la masa medidas por la celda de carga.");

  // Vistas
  PhyphoxBleExperiment::View viewGraph;
  viewGraph.setLabel("Gráfica fuerza");

  PhyphoxBleExperiment::View viewForce;
  viewForce.setLabel("Fuerza");

  PhyphoxBleExperiment::View viewMass;
  viewMass.setLabel("Masa");

  //vista para la gráfica de masa
  PhyphoxBleExperiment::View viewMassGraph;
  viewMassGraph.setLabel("Gráfica masa");

  // Gráfico de fuerza
  PhyphoxBleExperiment::Graph graphF;
  graphF.setLabel("Fuerza");
  graphF.setUnitX("s");
  graphF.setUnitY("N");
  graphF.setLabelX("Tiempo t");
  graphF.setLabelY("Fuerza F");
  graphF.setChannel(1, 2);  // tiempo, fuerza

  // gráfico de masa
  PhyphoxBleExperiment::Graph graphM;
  graphM.setLabel("Masa");
  graphM.setUnitX("s");
  graphM.setUnitY("g");
  graphM.setLabelX("Tiempo t");
  graphM.setLabelY("Masa m");
  graphM.setChannel(1, 3);  // tiempo, masa

  // Valor fuerza
  PhyphoxBleExperiment::Value force;
  force.setLabel("F");
  force.setPrecision(3);
  force.setUnit("N");
  force.setColor("FFFFFF");
  force.setChannel(2);
  force.setXMLAttribute("size=\"2\"");

  // Valor masa
  PhyphoxBleExperiment::Value mass;
  mass.setLabel("m");
  mass.setPrecision(1);
  mass.setUnit("g");
  mass.setColor("FFFFFF");
  mass.setChannel(3);
  mass.setXMLAttribute("size=\"2\"");

  // Añadir elementos a vistas
  viewGraph.addElement(force);
  viewGraph.addElement(graphF);

  viewForce.addElement(force);

  viewMass.addElement(mass);

  //añadir gráfica de masa
  viewMassGraph.addElement(mass);
  viewMassGraph.addElement(graphM);

  // Añadir vistas al experimento
  exp.addView(viewGraph);
  exp.addView(viewForce);
  exp.addView(viewMass);
  exp.addView(viewMassGraph); 

  PhyphoxBLE::addExperiment(exp);

  // --- Configuración HX711 ---
  LoadCell.begin();
  LoadCell.start(2000);
  LoadCell.setCalFactor(calFactor);

  pinMode(BUTTON_PIN, INPUT_PULLUP);
}

void loop() {

  // --- Tara con botón ---
  if (digitalRead(BUTTON_PIN) == LOW) {
    LoadCell.tareNoDelay();
    while (!LoadCell.getTareStatus()) {
      LoadCell.update();
    }
  }

  // --- Lectura del HX711 ---
  if (LoadCell.update()) {

    float t = (millis() - t_offset) / 1000.0;  // tiempo en segundos
    float grams = LoadCell.getData();          // masa en gramos
    float forceN = grams * 9.81 / 1000.0;      // fuerza en Newtons

    // Enviar a Phyphox: (t, fuerza, masa)
    PhyphoxBLE::write(t, forceN, grams);

    // Enviar por serie
    Serial.print(t);
    Serial.print(",");
    Serial.print(forceN, 3);
    Serial.print(",");
    Serial.println(grams, 1);
  }
}

void newExperimentEvent() {
  // Reiniciar tiempo al iniciar o reiniciar experimento
  if (PhyphoxBLE::eventType == 1 || PhyphoxBLE::eventType == 2) {
    t_offset = millis();
  }
}