/*Proyecto para medir variaciones rápidas de luminosidad como las producidas por la bombillas 
conectadas a la red eléctrica a 50 Hz.
Se necesita un fototransistor como el TEMT6000 y una placa ESP32 que dispone de bluetooth.
Este sensor ya está preparado para conectarse directamente a las entradas de las placas. 
El pin GND va a tierra (GND), el pin VCC a 3,3 o 5 V y el pin OUT va a la entrada 34.

Los datos obtenidos con este sketch solo se envian por bluetooth, no por el puerto serie.
Para ver los valores en un telefono móvil se ha de tener instalada la aplicación Phyphox
y dentro de ella el experimento "Fototransistor alta frecuencia" que se puede descargar desde
https://experimentaciolliure.wordpress.com/wp-content/uploads/2026/02/luz_paquetes10.zip
*/

#include <Arduino.h>
#include <BLEDevice.h>
#include <BLEServer.h>
#include <BLEUtils.h>
#include <BLE2902.h>

#define SERVICE_UUID        "19B10000-E8F2-537E-4F6C-D104768A1214"
#define CHARACTERISTIC_UUID "19B10001-E8F2-537E-4F6C-D104768A1214"

BLECharacteristic *pCharacteristic;
bool deviceConnected = false;
const int ldrPin = 34;

// Reducimos el bloque a 10 para garantizar cero pérdidas por Bluetooth
#define BATCH_SIZE 10 

struct SensorBatch {
  float tiempo[BATCH_SIZE];
  float valor[BATCH_SIZE];
};

QueueHandle_t dataQueue;

class MyServerCallbacks: public BLEServerCallbacks {
    void onConnect(BLEServer* pServer) { deviceConnected = true; }
    void onDisconnect(BLEServer* pServer) { deviceConnected = false; }
};

void adcTask(void *pvParameters) {
  SensorBatch batch;
  int index = 0;
  
  // Reloj estricto para no perder sincronía
  unsigned long nextSampleTime = micros(); 
  
  for(;;) {
    batch.tiempo[index] = micros() / 1000000.0;
    batch.valor[index] = (float)analogRead(ldrPin);
    index++;
    
    if (index >= BATCH_SIZE) {
      // Enviamos a la cola. Si está llena, descarta (pero con la nueva optimización no debería)
      xQueueSend(dataQueue, &batch, 0);
      index = 0;
    }
    
    // Calculamos matemáticamente el siguiente milisegundo exacto
    nextSampleTime += 1000;
    long waitTime = nextSampleTime - micros();
    
    if (waitTime > 0) {
      delayMicroseconds(waitTime);
    } else {
      nextSampleTime = micros(); // Resincronizar si hubo algún retraso grave
    }
  }
}

void bleTask(void *pvParameters) {
  SensorBatch batch;
  for(;;) {
    if (xQueueReceive(dataQueue, &batch, portMAX_DELAY)) {
      if (deviceConnected) {
        // Ahora enviamos 80 bytes, un tamaño mucho más digerible para BLE
        pCharacteristic->setValue((uint8_t*)&batch, sizeof(SensorBatch));
        pCharacteristic->notify();
      }
    }
  }
}

void setup() {
  Serial.begin(115200);
  
  // Aumentamos el tamaño de la sala de espera (cola a 50 bloques)
  dataQueue = xQueueCreate(50, sizeof(SensorBatch));

  BLEDevice::init("ESP32_Luz");
  BLEDevice::setMTU(256); // MTU intermedio seguro
  
  BLEServer *pServer = BLEDevice::createServer();
  pServer->setCallbacks(new MyServerCallbacks());
  
  BLEService *pService = pServer->createService(SERVICE_UUID);
  pCharacteristic = pService->createCharacteristic(
                      CHARACTERISTIC_UUID,
                      BLECharacteristic::PROPERTY_NOTIFY
                    );
  pCharacteristic->addDescriptor(new BLE2902());
  
  pService->start();
  BLEAdvertising *pAdvertising = BLEDevice::getAdvertising();
  pAdvertising->addServiceUUID(SERVICE_UUID);
  pServer->getAdvertising()->start();

  // Le damos mayor prioridad a la tarea de Bluetooth (2) para vaciar la cola rápido
  xTaskCreatePinnedToCore(adcTask, "Lectura", 4096, NULL, 1, NULL, 1);
  xTaskCreatePinnedToCore(bleTask, "Envio", 4096, NULL, 2, NULL, 0);
}

void loop() {
  vTaskDelete(NULL); 
}